﻿namespace Hims.Api.Controllers
{
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.PediatricGraphs;
    using Utilities;

    /// <inheritdoc />
    [Route("api/pediatric-chart")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PediatricChartController : BaseController
    {
        /// <summary>
        /// The pediatric chart service.
        /// </summary>
        private readonly IPediatricChartService pediatricChartService;

        /// <inheritdoc />
        public PediatricChartController(IPediatricChartService pediatricChartService) => this.pediatricChartService = pediatricChartService;

        /// <summary>
        /// Modifies the pediatric authority asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [Authorize]
        [HttpPost]
        [Route("modify-authority")]
        public async Task<ActionResult> ModifyPediatricAuthorityAsync([FromBody] PediatricChartAuthorityModel model)
        {
            model = (PediatricChartAuthorityModel)EmptyFilter.Handler(model);
            var response = await this.pediatricChartService.ModifyPediatricAuthority(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the pediatric authority asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [Authorize]
        [HttpPost]
        [Route("fetch-authority")]
        public async Task<ActionResult> FetchPediatricAuthorityAsync([FromBody] PediatricChartAuthorityModel model)
        {
            model ??= new PediatricChartAuthorityModel();
            var response = await this.pediatricChartService.FetchAuthorityAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Changes the pediatric authority status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [Authorize]
        [HttpPost]
        [Route("change-authority-status")]
        public async Task<ActionResult> ChangePediatricAuthorityStatusAsync([FromBody] PediatricChartAuthorityModel model)
        {
            model ??= new PediatricChartAuthorityModel();
            var response = await this.pediatricChartService.ChangeAuthorityStatusAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the pediatric chart type asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("fetch-chart-type")]
        public async Task<ActionResult> FetchPediatricChartTypeAsync()
        {
            var response = await this.pediatricChartService.FetchPediatricChartType();
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the pediatric display model.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("pediatric-display")]
        public async Task<ActionResult> FetchPediatricDisplayModel()
        {
            return this.Success(await this.pediatricChartService.FetchPediatricDisplayAsync());
        }

        /// <summary>
        /// Fetches the pediatric age wise data asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-age-wise-data")]
        public async Task<ActionResult> FetchPediatricAgeWiseDataAsync([FromBody] PediatricAgeWiseDataModel model)
        {
            model ??= new PediatricAgeWiseDataModel();
            var response = await this.pediatricChartService.FetchAgeWiseDataAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the pediatric age wise data z score asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-age-wise-z-score-data")]
        public async Task<ActionResult> FetchPediatricAgeWiseDataZScoreAsync([FromBody]PediatricAgeWiseDataZScoreModel model)
        {
            model ??= new PediatricAgeWiseDataZScoreModel();
            var response = await this.pediatricChartService.FetchAgeWiseDataZScoreAsync(model);
            return this.Success(response);
        }
    }
}